<?xml version="1.0" encoding="UTF-8"?>
<!-- 
    NAME: Card titles and contents v1_03.xsl
    LAST MODIFIED: 3/?/08
    
    PURPOSE: To transform an InfoML 0.88 document into a text file formatted as
    an in-order listing of all the cards in the document; the transformation is the same
    regardless of whether the original document is a list or an outline of infocards.
    This transformation file is to be used in conjunction with Infocard Organizer's Export
    function. Card titles and contents v1_01.xsl is known to work with the Infocard Organizer
    version dated 070813.
    
    BUGS:
    
    * none known; probably crashes if input file is invalid
    
    NOTES:
    
    * This program can be used as a template for XSLT transformations that
    visit every infocard in the order that they display in when viewed by
    Infocard Organizer. Change the section below marked "CASE 1" to specify
    how each infocard in turn is processed. The infocard currently being
    examined is in the variable $make. For example,
    
    * This transformation assumes that the first infoml element in the file
    contains nothing but a list of pointers to the top-level infocards. This
    is part of the design that enables an infocard file to represent either a
    list of infocards or an outline.
    
    * This transformation was designed to work with XSLT V1.0. It *IS*
    namespace-aware.
    -->
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="1.0"
	xmlns:q="http://infoml.org/infomlFile">

	<xsl:output media-type="text" omit-xml-declaration="yes"/>

	<xsl:template match="infomlFile">
		<xsl:apply-templates/>
	</xsl:template>

	<!-- This forces processor to ignore all infoml elements (except first one, as
		defined by the next template (below) -->
	<xsl:template match="q:infoml"/>

	<!-- This forces processor to process first infoml element. It begins by calling
		the "recurse" template on the root infoml element. -->
	<xsl:template match="//q:infoml[1]">
		<xsl:call-template name="recurse">
			<xsl:with-param name="node" select="."/>
			<xsl:with-param name="indentString" select="''"/>
		</xsl:call-template>
	</xsl:template>

	<!-- the "recurse" template -->
	<!-- Arguments: $node = node to be printed -->
	<!--     $indentString = contains spaces only; used to create indentation -->
	<xsl:template name="recurse">
		<xsl:param name="node"/>
		<xsl:param name="indentString"/>

		<!-- indentAmount controls the size of the indentation -->
		<xsl:variable name="indentAmount" select="'    '"/>

		<xsl:choose>
			<!-- CASE 1: This is a leaf node; print it -->
			<xsl:when test="$node/q:data/q:title or $node/q:data/q:content">
				<xsl:text>------ </xsl:text>
				
				<xsl:choose>
					<xsl:when test="$node/q:data/q:title != ''">
						<xsl:value-of select="$node/q:data/q:title"/>
					</xsl:when>
					<xsl:otherwise>
						<xsl:text>(no card title)</xsl:text>
					</xsl:otherwise>
				</xsl:choose>
				<xsl:text> ------</xsl:text>
				<xsl:text>&#10;&#10;</xsl:text>
				<xsl:for-each select="$node/q:data/q:content/q:p">
					<xsl:value-of select="normalize-space(.)"/>
					<xsl:text>&#10;&#10;</xsl:text>
				</xsl:for-each>
				<xsl:text>TAGS: </xsl:text>
				<xsl:for-each select="$node/q:selectors/q:tag">
					<xsl:value-of select="normalize-space(.)"/>
					<xsl:if test="position() != last()">
						<xsl:text>; </xsl:text>
					</xsl:if>
				</xsl:for-each>
				<xsl:text>&#10;&#10;&#10;&#10;</xsl:text>
			</xsl:when>

			<!-- CASE 2: This is a group node, with parent and child nodes -->
			<xsl:otherwise>
				<!-- Follow parentPtr to print the node it points to. -->
				<!-- parentNode is the actual infoml element to be printed. -->
				<xsl:for-each select="$node//q:parentPtr">
					<xsl:variable name="idOfParentNode" select="string (@targetId)"/>
					<xsl:variable name="parentNode" select="//q:infoml [@cardId = $idOfParentNode]"/>
					<xsl:call-template name="recurse">
						<xsl:with-param name="node" select="$parentNode"/>
						<xsl:with-param name="indentString" select="$indentString"/>
					</xsl:call-template>
				</xsl:for-each>
				<!-- Follow ptr to print the (child) node. -->
				<!-- childNode is the actual infoml element to be processed; recursion is necessary
				because childNode may itself be a parent/child pointer node. -->
				<xsl:for-each select="$node//q:ptr">
					<xsl:variable name="idOfChildNode" select="string (@targetId)"/>
					<xsl:variable name="childNode" select="//q:infoml [@cardId = $idOfChildNode]"/>
					<xsl:variable name="pointersOnlyNode" select="/q:infomlFile/q:infoml [1]"/>
					<xsl:choose>
						<xsl:when test="$node = $pointersOnlyNode">
							<xsl:call-template name="recurse">
								<xsl:with-param name="node" select="$childNode"/>
								<xsl:with-param name="indentString" select="$indentString"/>
							</xsl:call-template>
						</xsl:when>
						<xsl:otherwise>
							<xsl:call-template name="recurse">
								<xsl:with-param name="node" select="$childNode"/>
								<xsl:with-param name="indentString"
									select="concat($indentString, $indentAmount)"/>
							</xsl:call-template>
						</xsl:otherwise>
					</xsl:choose>

				</xsl:for-each>
			</xsl:otherwise>
		</xsl:choose>

	</xsl:template>


	<!-- This template "gobbles up" all text elements found between elements (mostly Return characters). -->
	<xsl:template match="text()"/>



</xsl:stylesheet>
